/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- *//* 
 * Copyright (C) 1998-2000 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Eskil Heyn Olsen
 *          Vadim Strizhevsky
 */

#include <glib.h>
#include <gnome.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-file.h>
#include <pi-dlp.h>
#include <pi-version.h>

#include <gpilotd/gnome-pilot-conduit-standard.h>

#include "time-conduit.h"

#define TC_DEBUG 1

#ifdef TC_DEBUG
#define LOG(args...) g_log (G_LOG_DOMAIN, \
                            G_LOG_LEVEL_MESSAGE, \
                             "time_file: "##args)
#else
#define LOG(args...)
#endif

GnomePilotConduit *conduit_load_gpilot_conduit (GPilotPilot *pilot);
void conduit_destroy_gpilot_conduit (GnomePilotConduit *conduit);

#define CONDUIT_VERSION "0.1"

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar *prefix;
	gchar *buf;
	g_return_if_fail(c!=NULL);
	
	prefix = g_strdup_printf (CONFIG_PREFIX, pilotId);
 
	*c = g_new0(ConduitCfg,1);
	gnome_config_push_prefix(prefix);

	(*c)->sync_type = GnomePilotConduitSyncTypeCustom; /* this will be reset by capplet */
	(*c)->direction = gnome_config_get_int ("direction=1"); 
	gnome_config_pop_prefix();

	(*c)->pilotId = pilotId;
	
	g_free (prefix);
}

static void 
copy_configuration(ConduitCfg *d, ConduitCfg *c)
{
	g_return_if_fail(c!=NULL);
	g_return_if_fail(d!=NULL);
	d->direction = c->direction;
	d->pilotId = c->pilotId;
}

static ConduitCfg*
dupe_configuration(ConduitCfg *c) 
{
	ConduitCfg *d;
	g_return_val_if_fail(c!=NULL,NULL);
	d = g_new0(ConduitCfg,1);
	copy_configuration(d,c);
	return d;
}

static void 
destroy_configuration(ConduitCfg **c) 
{
	g_return_if_fail(c!=NULL);
	g_free(*c);
	*c = NULL;
}

static void 
save_configuration(ConduitCfg *c) 
{
	gchar *prefix;

	g_return_if_fail(c!=NULL);
	prefix = g_strdup_printf (CONFIG_PREFIX, c->pilotId);

	gnome_config_push_prefix(prefix);
	gnome_config_set_int ("direction", c->direction);
	gnome_config_pop_prefix();
	gnome_config_sync();
	gnome_config_drop_all();
	g_free (prefix);
}

void set_to_sync_to_pc (GtkMenuItem *item,
			ConduitCfg *cfg)
{
	cfg->direction = PDA_TO_PC;
}

void set_to_sync_to_pda (GtkMenuItem *item,
			 ConduitCfg *cfg)
{
	cfg->direction = PC_TO_PDA;
}

static GtkWidget
*createCfgWindow(GnomePilotConduit* conduit, ConduitCfg *cfg)
{
	GtkWidget *vbox;
	GtkWidget *option;
	GtkMenu *menu, *m1, *m2;
	GtkWidget *label;

	vbox = gtk_vbox_new (FALSE, GNOME_PAD);	

	menu = gtk_menu_new ();

	m1 = gtk_menu_item_new_with_label (_("Update PDA time"));
	gtk_signal_connect (GTK_OBJECT (m1), "activate", 
			    GTK_SIGNAL_FUNC (set_to_sync_to_pda), cfg);
	gtk_menu_append (menu, GTK_WIDGET (m1));

	m2 = gtk_menu_item_new_with_label (_("Update PC time"));
	gtk_signal_connect (GTK_OBJECT (m2), "activate",
			    GTK_SIGNAL_FUNC (set_to_sync_to_pc), cfg);
	gtk_menu_append (menu, GTK_WIDGET (m2));

	option = gtk_option_menu_new ();
	gtk_option_menu_set_menu (GTK_OPTION_MENU (option), GTK_WIDGET (menu));

	gtk_box_pack_start (GTK_BOX (vbox), option, FALSE, FALSE, GNOME_PAD);
	
	label = gtk_label_new (_("Please note, that some versions of\n"
				 "PalmOS<tm> may crash the PDA if the\n"
				 "PDA's time is modifiedduring synchronization.\n"
				 "\n"
				 "A later version of this capplet will probably\n"
				 "be able to check your PDA's OS version for this,\n"
				 "but for now, it's trial and error."));
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, GNOME_PAD);

	return vbox;
}


static void
setOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
}


static void
readOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
}


static gint
create_settings_window (GnomePilotConduit *conduit, GtkWidget *parent, gpointer data)
{
	GtkWidget *cfgWindow;
	LOG("create_settings_window");

	cfgWindow = createCfgWindow(conduit, GET_CONDUIT_CFG(conduit));

	gtk_container_add(GTK_CONTAINER(parent),cfgWindow);
	gtk_widget_show_all(cfgWindow);

	gtk_object_set_data(GTK_OBJECT(conduit),OBJ_DATA_CONFIG_WINDOW,cfgWindow);
	setOptionsCfg(GET_CONDUIT_WINDOW(conduit),GET_CONDUIT_CFG(conduit));
	return 0;
}
static void
display_settings (GnomePilotConduit *conduit, gpointer data)
{
	LOG("display_settings");
	setOptionsCfg(GET_CONDUIT_WINDOW(conduit),GET_CONDUIT_CFG(conduit));
}

static void
save_settings    (GnomePilotConduit *conduit, gpointer data)
{
	LOG("save_settings");
	readOptionsCfg(GET_CONDUIT_WINDOW(conduit),GET_CONDUIT_CFG(conduit));
	save_configuration(GET_CONDUIT_CFG(conduit));
}

static void
revert_settings  (GnomePilotConduit *conduit, gpointer data)
{
	ConduitCfg *cfg,*cfg2;
	LOG("revert_settings");
	cfg2= GET_CONDUIT_OLDCFG(conduit);
	cfg = GET_CONDUIT_CFG(conduit);
	save_configuration(cfg2);
	copy_configuration(cfg,cfg2);
	setOptionsCfg(GET_CONDUIT_WINDOW(conduit),cfg);
}

static gint
synchronize (GnomePilotConduit *c,
	     GnomePilotDBInfo *dbi)
{
	ConduitCfg *cfg;

	cfg = GET_CONDUIT_CFG (c);

	if (cfg->direction == PDA_TO_PC) {
		int err;
		err = dlp_GetSysDateTime (dbi->pilot_socket, &(cfg->t));
		LOG ("synchronization to PC = %d, err = %d", cfg->t, err);
	} else {
		time_t t;
		int err;
		t = time (NULL);
		err = dlp_SetSysDateTime (dbi->pilot_socket, t);
		LOG ("synchronization to PDA = %d", t, err);
	}
}

GnomePilotConduit *
conduit_load_gpilot_conduit (GPilotPilot *pilot)
{
	GtkObject *retval;
	ConduitCfg *cfg, *cfg2;
	
	retval = gnome_pilot_conduit_standard_new ("Unsaved Preferences", (guint32)0x70737973, pilot);
	g_assert (retval != NULL);
	
	LOG("creating time conduit");

	load_configuration (&cfg, pilot->pilot_id);
	cfg2 = dupe_configuration (cfg);

	LOG ("sync dir = %p->%d", cfg, cfg->direction);

	gtk_object_set_data(GTK_OBJECT(retval),OBJ_DATA_CONFIG,cfg);
	gtk_object_set_data(GTK_OBJECT(retval),OBJ_DATA_OLDCONFIG,cfg2);

	gtk_signal_connect (retval, GNOME_PILOT_CONDUIT_STANDARD_SIGNAL_SYNCHRONIZE (synchronize), cfg);
	gtk_signal_connect (retval, GNOME_PILOT_CONDUIT_SIGNAL_CREATE_SETTINGS_WINDOW (create_settings_window), NULL);
	gtk_signal_connect (retval, GNOME_PILOT_CONDUIT_SIGNAL_DISPLAY_SETTINGS (display_settings), NULL);
	gtk_signal_connect (retval, GNOME_PILOT_CONDUIT_SIGNAL_SAVE_SETTINGS (save_settings), NULL);
	gtk_signal_connect (retval, GNOME_PILOT_CONDUIT_SIGNAL_REVERT_SETTINGS (revert_settings), NULL);

	return GNOME_PILOT_CONDUIT (retval);
}

void
conduit_destroy_gpilot_conduit (GnomePilotConduit *conduit)
{
	ConduitCfg  *cfg=GET_CONDUIT_CFG(conduit);
	ConduitCfg  *cfg2=GET_CONDUIT_OLDCFG(conduit);
	LOG("destroying time conduit");
	
	destroy_configuration(&cfg);
	if(cfg2) destroy_configuration(&cfg2);
	gtk_object_destroy (GTK_OBJECT (conduit));
}


