/***************************************************************************
	kjvis.cpp  -  Visualizations used in the KJfol-GUI
	--------------------------------------
	Maintainer: Stefan Gehn <sgehn@gmx.net>

 ***************************************************************************/

// local includes
#include "kjvis.h"

// system includes
#include <math.h>

//qt includes
#include <qpainter.h>
#include <qsize.h>

//kde includes
#include <kglobal.h>
#include <kconfig.h>
#include <kpixmapeffect.h>
#include <kpixmap.h>

// noatun includes
#include <noatun/player.h>

/*******************************************
 * KJVisScope
 *******************************************/

void KJVisScope::swapScope(Visuals newOne)
{
	QStringList line = parent()->item("analyzerwindow");
	KJLoader *p=parent();
	p->removeChild(this);
	delete this;

	KGlobal::config()->setGroup("KJofol-Skins");
	KGlobal::config()->writeEntry("AnalyzerType", newOne);
	KGlobal::config()->sync();

	KJWidget *w;
	switch (newOne)
	{
	case Null:
		w=new KJNullScope(line, p);
		break;
	case FFT:
		w=new KJVis(line, p);
		break;
	case Mono:
		w=new KJScope(line, p);
	};

	p->addChild(w);
}

/*******************************************
 * KJNullScope
 *******************************************/

KJNullScope::KJNullScope(const QStringList &l, KJLoader *parent)
	: KJVisScope(parent)
{
	int x  = l[1].toInt();
	int y  = l[2].toInt();
	int xs = l[3].toInt() - x;
	int ys = l[4].toInt() - y;

	// background under vis
	QPixmap tmp  = parent->pixmap(parent->item("backgroundimage")[1]);
	mBack = new KPixmap ( QSize(xs,ys) );
	bitBlt( mBack, 0, 0, &tmp, x, y, xs, ys, Qt::CopyROP );

	setRect ( x, y, xs, ys );

	repaint();
}

void KJNullScope::paint(QPainter *p, const QRect &)
{
	// just redraw the background
	bitBlt ( p->device(), rect().topLeft(), mBack, QRect(0,0,-1,-1), Qt::CopyROP );
}

bool KJNullScope::mousePress(const QPoint &)
{
	parent()->repaint(rect(), false);
	swapScope(FFT);

	return true;
}


/*******************************************
 * KJVis
 *******************************************/

KJVis::KJVis(const QStringList &l, KJLoader *parent)
	: KJVisScope(parent), MonoFFTScope(30), mGradient(0)
{
	int x=l[1].toInt();
	int y=l[2].toInt();
	int xs=l[3].toInt()-x;
	int ys=l[4].toInt()-y;

	if ( parent->exist("analyzercolor") )
	{
		QStringList &col = parser()["analyzercolor"];
		mColor.setRgb ( col[1].toInt(), col[2].toInt(), col[3].toInt() );
	}
	else // TODO: what should be default colors for Vis?
	{
		mColor.setRgb ( 255, 255, 255 ); // white is default
	}

	// background under vis
	QPixmap tmp  = parent->pixmap(parent->item("backgroundimage")[1]);
	mBack = new KPixmap ( QSize(xs,ys) );
	bitBlt( mBack, 0, 0, &tmp, x, y, xs, ys, Qt::CopyROP );

	mAnalyzer = new KPixmap ( QSize(xs,ys) );
	bitBlt( mAnalyzer, 0, 0, &tmp, x, y, xs, ys, Qt::CopyROP );

	// create a gradient for the bars going from 30% lighter to 30% darker than mColor
	mGradient = new KPixmap ( QSize(xs,ys) );
	KPixmapEffect::gradient ( *mGradient, mColor.light(130), mColor.dark(130), KPixmapEffect::VerticalGradient );

	setRect (x,y,xs,ys);

	// each bar will be 1px wide
	mMultiples=1;
	setBands(magic(xs/mMultiples));
	start();
}

void KJVis::scopeEvent(float *d, int size)
{
	if ( !napp->player()->isPlaying() ) // don't draw if we aren't playing (either paused or stopped)
	{
		if ( napp->player()->isStopped() ) // clear vis-window if playing has been stopped
			parent()->repaint(rect(), false);
		return;
	}

	int x = 0 /*rect().x()*/;
	int h = rect().height();

	QBitmap mGradientMask ( rect().width(), h, true );
	QPainter mask( &mGradientMask );

	float *start = d ;
	float *end = d + size /*- 1*/;

	// loop creating the mask for vis-gradient
	for ( ; start < end; ++start )
	{
		// 5 has been 8 before and I have no idea how this scaling works :/
		// FIXME: somebody please make it sclae to 100% of height,
		//        I guess that would be nicer to look at
//		float n = log((*start)+1) * (float)h * 5;
		float n = log((*start)+1) * (float)h * 5;
		int amp=(int)n;

		// range check
		if ( amp < 0 ) amp = 0;
		else if ( amp > h ) amp = h;

		// make a part of the analyzer-gradient visible
		mask.fillRect ( x, (h-amp), mMultiples, amp, Qt::color1 );
		x += mMultiples;
	}
	// done creating our mask

/* ORIGINAL CODE for drawing vis
	{
		float n=(*start)+1;
		n=log(n)*fheight*8;
		int amp=(int)n;
		if (amp<0) amp=0;
		if (amp>h) amp=h;
		p.fillRect(x, y+(h-amp), mMultiples, amp, mColor);
		x += mMultiples;
	}
*/

	// temp pixmap to avoid flicker
//	QPixmap temp (  rect().width(), rect().height() );
	// draw background of vis into it
	bitBlt ( mAnalyzer, 0, 0, mBack, 0, 0, -1, -1, Qt::CopyROP );

	// draw the analyzer
	mGradient->setMask(mGradientMask);
	bitBlt ( mAnalyzer, 0, 0, mGradient, 0, 0, -1, -1, Qt::CopyROP );

	// put that thing on screen
//	QPainter p ( parent() );
//	p.drawPixmap ( rect().x(), rect().y(), temp );

	repaint();
}

void KJVis::paint(QPainter *p, const QRect &)
{
	// put that thing on screen
	if ( !napp->player()->isStopped() )
		bitBlt ( p->device(), rect().topLeft(), mAnalyzer, QRect(0,0,-1,-1), Qt::CopyROP );
}


bool KJVis::mousePress(const QPoint &)
{
	stop();
	parent()->repaint(rect(), false);
	swapScope(Mono);
	return true;
}


/*******************************************
 * KJScope - an oscilloscope drawn into the skin
 *******************************************/

KJScope::KJScope(const QStringList &l, KJLoader *parent)
	: KJVisScope(parent), MonoScope(30), blurnum(0), mOsci(0)
{
	int x=l[1].toInt();
	int y=l[2].toInt();
	int xs = mWidth = l[3].toInt()-x;
	int ys = mHeight = l[4].toInt()-y;

//	kdDebug(66666) << "Analyzer Window " << x << "," << y << " " << mWidth << "," << mHeight << endl;

	if ( parent->exist("analyzercolor") )
	{
		QStringList &col = parser()["analyzercolor"];
		mColor.setRgb ( col[1].toInt(), col[2].toInt(), col[3].toInt() );
	}
	else // FIXME: what should be default colors for Vis?
		mColor.setRgb ( 255, 255, 255 );

	// background under vis
	QPixmap tmp  = parent->pixmap(parent->item("backgroundimage")[1]);
	mBack = new KPixmap ( QSize(xs,ys) );
	bitBlt( mBack, 0, 0, &tmp, x, y, xs, ys, Qt::CopyROP );

	mOsci = new KPixmap ( QSize(xs,ys) );
	// fill our buffer with valid pixels, i.e. vis-background
	bitBlt( mOsci, 0, 0, &tmp, x, y, xs, ys, Qt::CopyROP );

	setRect ( x, y, xs, ys );

	// set the samplewidth to the largest integer divisible by mWidth
	setSamples ( xs );

	start();
}

void KJScope::scopeEvent(float *d, int size)
{
	if ( !napp->player()->isPlaying() )
	{
		if ( napp->player()->isStopped() )
		{
			bitBlt ( mOsci, 0, 0, mBack, 0, 0, -1, -1, Qt::CopyROP );
			repaint();
		}
		return;
	}

	float *start = d;
	float *end = d + size;

	int heightHalf = rect().height()/2 /* -1 */;
	int x = 0;

//	QPixmap temp( rect().width(), rect().height() );
	QPainter tempP( mOsci );

//	tempP.setPen( mColor );

	if ( blurnum == 3 )
	{  // clear whole Vis
		bitBlt ( mOsci, 0, 0, mBack, 0, 0, -1, -1, Qt::CopyROP );
		tempP.setPen( mColor.light(110) ); // 10% lighter than mColor
		blurnum=0;
	}
	else
	{
		blurnum++;
		// reduce color for blur-effect
		tempP.setPen( mColor.dark(90+(10*blurnum)) ); // darken color
	}

	for ( ; start < end; ++start )
	{
		float n = (*start) * (float)heightHalf;
		int amp = (int)n;

		// range check
		if ( amp > heightHalf ) amp = heightHalf;
		else if ( amp < -heightHalf) amp = -heightHalf;

		// draw
		tempP.drawLine(x, heightHalf, x, heightHalf+amp);
		x++;
	}

	// put that thing on screen
//	QPainter p ( parent() );
//	p.drawPixmap ( rect().x(), rect().y(), *mOsci );
	repaint();
}

void KJScope::paint(QPainter *p, const QRect &)
{
	// put that thing on screen
//	if ( !napp->player()->isStopped() )
		bitBlt ( p->device(), rect().topLeft(), mOsci, QRect(0,0,-1,-1), Qt::CopyROP );
}

bool KJScope::mousePress(const QPoint &)
{
	stop();
	parent()->repaint(rect(), false);
	swapScope(Null);
	return true;
}
