/* test program for KMime::Codec's:
   compile with:
   g++ -I$QTDIR/include -I$KDEDIR/include -L$QTDIR/lib -L$KDEDIR/lib \
   -lqt-mt -lkdecore -lkdenetwork -O2 -pthread -DQT_THREAD_SUPPORT \
   -o test_kmime_codec_encoder{,.cpp}
*/

// return codes:
#define USAGE_DISPLAYED 1
#define UNKNOWN_CODEC 2
#define INFILE_READ_ERR 3
#define OUTFILE_WRITE_ERR 4

#include <../kmime_codecs.h>

#include <kdebug.h>

#include <cstdlib>
#include <iostream>

#define _GNU_SOURCE
#include <getopt.h>
#include <assert.h>

#include <qfile.h>
#include <qcstring.h> // QByteArray

using namespace KMime;
using namespace std;

static struct option long_options[] = {
  { "encode", 1, 0, 0 },
  { "decode", 1, 0, 0 },
  { "output-buffer-size", 1, 0, 0 },
  { "input-buffer-size", 1, 0, 0 },
  { "outfile", 1, 0, 0 },
  { "with-crlf", 0, 0, 0},
  { "iterations", 1, 0, 0},
  { "without-finish", 0, 0, 0},
  { 0, 0, 0, 0 }
};

void usage() {
  cerr << "usage: test_kmime_codec (--encode|--decode) "
    "<encoding-name> [options] infile\n"
    "where options include:\n\n"
    " --outfile <outfile>          write output into file <outfile>\n"
    " --output-buffer-size <size>  en/decode into chunks of <size> bytes\n"
    "                              default: 4096\n"
    " --input-buffer-size <size>   en/decode from chunks of ,size> bytes\n"
    "                              default: slurp in whole file\n"
    " --with-crlf                  use CRLF instead of LF in output\n"
    " --iterations <number>        do more than one iteration\n"
    "                              default: 1\n"
    " --without-finish             don't call the finish() method\n"
       << endl;
  exit(USAGE_DISPLAYED);
}

int main( int argc, char * argv[] ) {

  int outbufsize = 4096;
  int inbufsize = -1; // whole file
  int iterations = 1;
  bool encode = false;
  bool decode = false;
  bool writing = false;
  bool withCRLF = false;
  bool withFinish = true;
  QCString outfilename, infilename;
  QCString encodingName;

  // options parsing:
  while( 1 ) {
    int option_index = 0;
    int c = getopt_long( argc, argv, "", long_options, &option_index );
    if ( c == -1 ) break;
    if ( c == 0 ) {
      switch ( option_index ) {
      case 0: // encode
	if ( decode || !optarg || !*optarg ) usage();
	encode = true;
	encodingName = QCString(optarg);
	break;
      case 1: // decode
	if ( encode || !optarg || !*optarg ) usage();
	decode = true;
	encodingName = QCString(optarg);
	break;
      case 2: // output-buffer-size
	if ( !optarg || (outbufsize = atoi( optarg )) < 1 )
	  usage();
	break;
      case 3: // input-buffer-size
	if ( !optarg || (inbufsize = atoi( optarg )) < 1 )
	  usage();
	break;
      case 4: // outfile
	if ( !optarg || !*optarg ) usage();
	outfilename = QCString(optarg);
	writing = true;
	break;
      case 5: // with-crlf
	withCRLF = true;
	break;
      case 6: // iterations
	if ( !optarg || (iterations = atoi( optarg )) < 1 )
	  usage();
      case 7: // without-finish
	withFinish = false;
	break;
      default: usage();
      }
    }
  }

  if ( !decode && !encode ) usage();
  if ( decode && encode ) usage();

  if ( encode )
    kdDebug() << "encoding as " << encodingName << endl;
  else if ( decode )
    kdDebug() << "decoding " << encodingName << endl;

  if ( optind != argc - 1 ) usage();

  QFile infile( argv[ optind ] );
  if (!infile.exists()) {
    kdDebug() << "infile \"" << infile.name() << "\" does not exist!" << endl;
    return INFILE_READ_ERR;
  }
  if (!infile.open( IO_ReadOnly )) {
    kdDebug() << "cannot open " << infile.name() << " for reading!"
	      << endl;
    return INFILE_READ_ERR;
  }

  QFile outfile( outfilename );
  if ( !outfilename.isEmpty() ) {
    if (!outfile.open( IO_WriteOnly|IO_Truncate )) {
      kdDebug() << "cannot open " << outfile.name() << " for writing!"
		<< endl;
      return OUTFILE_WRITE_ERR;
    }
  }

  if ( inbufsize < 0 )
    inbufsize = infile.size();

  QByteArray indata( inbufsize );
  QByteArray outdata( outbufsize );

  kdDebug() << "using output buffer size of " << outbufsize << endl;
  kdDebug() << "using  input buffer size of " << inbufsize << endl;
  if ( !withFinish )
    kdWarning() << "omitting finish calls. Results may be truncated!" << endl;

  // get a codec. Don't delete it later!!
  Codec<> * codec = Codec<>::codecForName( encodingName );
  if ( !codec ) {
    kdDebug() << "unknown codec \"" << encodingName << "\"" << endl;
    return UNKNOWN_CODEC;
  }

  Encoder<> * enc = 0;
  Decoder<> * dec = 0;

  // we're going to need this below:
#define write_full_outdata_then_reset  do { if ( writing ) { \
	      Q_LONG outlen = outfile.writeBlock( outdata.data(), \
						  outdata.size() ); \
	      if ( outlen != (int)outdata.size() ) \
		return OUTFILE_WRITE_ERR; \
	    } \
	    oit = outdata.begin(); \
          } while ( false )


  //
  // Loop over iterations:
  //
  for( int i = 0 ; i < iterations ; i++ ) {

    // Initialize the output iterators:
    QByteArray::Iterator oit = outdata.begin();
    QByteArray::Iterator oend = outdata.end();

    // Get an encoder. This one you have to delete!
    if ( encode ) {
      enc = codec->makeEncoder( withCRLF );
      assert( enc );
    } else {
      dec = codec->makeDecoder( withCRLF );
      assert( dec );
    }

    //
    // Loop over input chunks:
    //
    Q_LONG reallyRead = 0;
    while ( (reallyRead = infile.readBlock( indata.data(), indata.size() )) ) {
      
      // setup input iterators:
      QByteArray::Iterator iit = indata.begin();
      QByteArray::Iterator iend = indata.begin() + reallyRead;

      if ( encode )
	//
	// Loop over encode() calls:
	//
	while ( !enc->encode( iit, iend, oit, oend ) )
	  if ( oit == oend )
	    // output buffer full:
	    write_full_outdata_then_reset;
      else
	//
	// Loop over decode() calls:
	//
	while ( !dec->decode( iit, iend, oit, oend ) )
	  if ( oit == oend )
	    // output buffer full:
	    write_full_outdata_then_reset;
    } // end loop over input chunks

    //
    // Now finish the encoding/decoding:
    // (same loops as above, just s/encode|decode/finish())
    //
    if ( withFinish )
      if ( encode )
	while ( !enc->finish( oit, oend ) )
	  if ( oit == oend )
	    write_full_outdata_then_reset;
      else
	while ( !dec->finish( oit, oend ) )
	  if ( oit == oend )
	    write_full_outdata_then_reset;

    //
    // Write out last (partial) output chunk:
    //
    if ( writing ) {
      Q_LONG outlen = outfile.writeBlock( outdata.data(),
					  oit - outdata.begin() );
      if ( outlen != oit - outdata.begin() )
	return OUTFILE_WRITE_ERR;
    }

    //
    // Delete en/decoder:
    //
    if ( encode )
      delete enc;
    else
      delete dec;
  }

  return 0;
}
