#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "interface.h"
#include "support.h"

#include "waterfall.h"
#include "gtkdial.h"
#include "miniscope.h"
#include "papertape.h"

#include "main.h"
#include "conf.h"
#include "ascii.h"
#include "trx.h"
#include "macro.h"
#include "log.h"

GtkWidget *appwindow;
GtkWidget *WFPopupMenu;
Waterfall *waterfall;

GtkTextTag *rxtag;
GtkTextTag *txtag;
GtkTextTag *hltag;

gchar *SoftString = "gMFSK v" VERSION " (Gnome MFSK) by OH2BNS";

/*
 * These get used a lot so lets have static pointers to them.
 */
static GtkScrolledWindow	*rxscrolledwindow;
static GtkSpinButton		*freqspinbutton;
static GtkDial			*metricdial;

static GtkTextView		*rxtextview;
static GtkTextView		*txtextview;
static GtkTextBuffer		*rxbuffer;
static GtkTextBuffer		*txbuffer;

/* ---------------------------------------------------------------------- */

static gboolean error_idle_func(gpointer data)
{
	GtkWidget *dialog;
	gchar *msg = (gchar *) data;

	gdk_threads_enter();

	dialog = gtk_message_dialog_new(GTK_WINDOW(appwindow),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE,
					"%s", msg);

	gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy (dialog);

	g_free(data);

	gdk_threads_leave();

	return FALSE;
}

void errmsg(const gchar *fmt, ...)
{
	va_list args;
	gchar *msg;

	va_start(args, fmt);
	msg = g_strdup_vprintf(fmt, args);
	va_end(args);

	g_idle_add(error_idle_func, (gpointer) msg);
}

/* ---------------------------------------------------------------------- */

void push_button(const gchar *name)
{
	GtkToggleButton *button;

	button = GTK_TOGGLE_BUTTON(lookup_widget(appwindow, name));
	gtk_toggle_button_set_active(button, TRUE);
}

/* ---------------------------------------------------------------------- */

static gboolean statusbar_set_mode_callback(gpointer data)
{
	static gint id = 0;
	GtkStatusbar *bar;
	gint mode;

	gdk_threads_enter();

	bar = GTK_STATUSBAR(lookup_widget(appwindow, "modestatusbar"));

	if (id == 0)
		id = gtk_statusbar_get_context_id(bar, "statusbar trx mode");

	if (id != 0) {
		mode = GPOINTER_TO_INT(data);

		gtk_statusbar_pop(bar, id);
		gtk_statusbar_push(bar, id, trx_mode_names[mode]);
	}

	gdk_threads_leave();
	return FALSE;
}

void statusbar_set_mode(gint mode)
{
	g_idle_add(statusbar_set_mode_callback, GINT_TO_POINTER(mode));
}

static gboolean statusbar_set_trxstate_callback(gpointer data)
{
	static gint id = 0;
	GtkStatusbar *bar;
	gint state;

	gdk_threads_enter();

	bar = GTK_STATUSBAR(lookup_widget(appwindow, "trxstatusbar"));

	if (id == 0)
		id = gtk_statusbar_get_context_id(bar, "statusbar trx state");

	if (id != 0) {
		state = GPOINTER_TO_INT(data);

		gtk_statusbar_pop(bar, id);
		gtk_statusbar_push(bar, id, trx_state_names[state]);
	}

	gdk_threads_leave();
	return FALSE;
}

void statusbar_set_trxstate(gint state)
{
	g_idle_add(statusbar_set_trxstate_callback, GINT_TO_POINTER(state));
}

/* ---------------------------------------------------------------------- */

void textbuffer_insert_end(GtkTextBuffer *buf, const gchar *str, int len)
{
	GtkTextIter end;
	gchar *s;

	gtk_text_buffer_get_end_iter(buf, &end);

	s = g_locale_to_utf8(str, len, NULL, NULL, NULL);
	gtk_text_buffer_insert(buf, &end, s, -1);
	g_free(s);
}

void textbuffer_delete_end(GtkTextBuffer *buffer, gint count)
{
	GtkTextIter start, end;

	gtk_text_buffer_get_end_iter(buffer, &start);
	gtk_text_buffer_get_end_iter(buffer, &end);
	gtk_text_iter_backward_chars(&start, count);
	gtk_text_buffer_delete(buffer, &start, &end);
}

void textview_scroll_end(GtkTextView *view)
{
	GtkTextBuffer *buffer;
	GtkTextMark *mark;

	buffer = gtk_text_view_get_buffer(view);
	mark = gtk_text_buffer_get_mark(buffer, "end");
	gtk_text_view_scroll_mark_onscreen(view, mark);
}

/* ---------------------------------------------------------------------- */

void send_char(gchar c)
{
	textbuffer_insert_end(txbuffer, &c, 1);
	textview_scroll_end(txtextview);

	trx_put_tx_char(c);
}

void send_string(const gchar *str)
{
	textbuffer_insert_end(txbuffer, str, -1);
	textview_scroll_end(txtextview);

	while (*str)
		trx_put_tx_char(*str++);
}

/* ---------------------------------------------------------------------- */

void insert_rx_text(gchar *tag, gchar *str, int len)
{
	GtkTextIter end;

	gtk_text_buffer_get_end_iter(rxbuffer, &end);

	str = g_locale_to_utf8(str, len, NULL, NULL, NULL);
	gtk_text_buffer_insert_with_tags_by_name(rxbuffer, &end,
						 str, len,
						 tag, NULL);
	g_free(str);
}

static gboolean main_loop(gpointer unused)
{
	static gboolean crflag = FALSE;
	static guchar databuf[30];
	static gint dataptr = 0;
	gboolean textflag = FALSE;
	gchar *p;
	gint c;

	gdk_threads_enter();

	/*
	 * Check for received papertape data.
	 */
	while ((c = trx_get_rx_data()) != -1) {
		Papertape *t;

		databuf[dataptr++] = c;

		if (dataptr < 30)
			continue;

		t = PAPERTAPE(lookup_widget(appwindow, "rxtape1"));
		c = papertape_setdata(t, databuf, 30);

		if (c == 0) {
			t = PAPERTAPE(lookup_widget(appwindow, "rxtape3"));
			p = papertape_getdata_full(t);
			t = PAPERTAPE(lookup_widget(appwindow, "rxtape4"));
			papertape_setdata_full(t, p);

			t = PAPERTAPE(lookup_widget(appwindow, "rxtape2"));
			p = papertape_getdata_full(t);
			t = PAPERTAPE(lookup_widget(appwindow, "rxtape3"));
			papertape_setdata_full(t, p);

			t = PAPERTAPE(lookup_widget(appwindow, "rxtape1"));
			p = papertape_getdata_full(t);
			t = PAPERTAPE(lookup_widget(appwindow, "rxtape2"));
			papertape_setdata_full(t, p);

			t = PAPERTAPE(lookup_widget(appwindow, "rxtape1"));
			papertape_clear(t);
		}

		dataptr = 0;
	}

	/*
	 * Check for received characters.
	 */
	while ((c = trx_get_rx_char()) != -1) {
		if (c == 0)
			continue;

		/* CR-LF reduced to a single LF */
		if (crflag && c == 10) {
			crflag = FALSE;
			continue;
		} else if (c == 13)
			crflag = TRUE;

		/* backspace is a special case */
		if (c == 8) {
			textbuffer_delete_end(rxbuffer, 1);
			log_to_file(LOG_RX, "<BS>");
			continue;
		}

		p = ascii[c];
		if (*p == '<' && c != '<')
			insert_rx_text("hltag", p, -1);
		else
			insert_rx_text("rxtag", p, -1);
		log_to_file(LOG_RX, p);

		textflag = TRUE;
	}

	/*
	 * Check for transmitted characters.
	 */
	while ((c = trx_get_echo_char()) != -1) {
		if (c == 0)
			continue;

		/* backspace is a special case */
		if (c == 8) {
			textbuffer_delete_end(rxbuffer, 1);
			log_to_file(LOG_TX, "<BS>");
			continue;
		}

		p = ascii[c];
		insert_rx_text("txtag", p, -1);
		log_to_file(LOG_TX, p);

		textflag = TRUE;
	}

	if (textflag == TRUE)
		textview_scroll_end(rxtextview);

	gtk_dial_set_value(metricdial, trx_get_metric());

	if (!GTK_WIDGET_HAS_FOCUS(freqspinbutton))
		gtk_spin_button_set_value(freqspinbutton, trx_get_freq());

	gdk_threads_leave();

	return TRUE;
}

/* ---------------------------------------------------------------------- */

static gchar *timestring(void)
{
	static gchar buf[64];
	struct tm *tm;
	time_t t;

	time(&t);
	tm = gmtime(&t);

	strftime(buf, sizeof(buf), "%H:%M:%SZ", tm);
	buf[sizeof(buf) - 1] = 0;

	return buf;
}

/* ---------------------------------------------------------------------- */

static gint clock_cb(gpointer ptr)
{
	GtkWidget *widget;

	gdk_threads_enter();

	widget = lookup_widget(appwindow, "clocklabel");
	gtk_label_set_text(GTK_LABEL(widget), timestring());

	gdk_threads_leave();

	return TRUE;
}

static void frequency_set_cb(GtkWidget *widget, gfloat freq, gpointer data)
{
	trx_set_freq(freq);
}

/* ---------------------------------------------------------------------- */

int main (int argc, char *argv[])
{
	GtkStatusbar *statusbar;
	GnomeUIInfo *uiinfo;
	GtkTextIter iter;
	GtkTextTag *tag;

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	g_thread_init(NULL);
	gdk_threads_init();

	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
			    argc, argv,
			    GNOME_PARAM_APP_DATADIR, PACKAGE_DATA_DIR,
			    NULL);

	/* create main window */
	appwindow = create_appwindow ();

	/* connect waterfall "frequency set" signal */
	waterfall = WATERFALL(lookup_widget(appwindow, "waterfall"));
	gtk_signal_connect(GTK_OBJECT(waterfall), "frequency_set",
			   GTK_SIGNAL_FUNC(frequency_set_cb),
			   NULL);

	/* create waterfall popup menu */
	WFPopupMenu = create_wfpopupmenu();

	rxscrolledwindow = GTK_SCROLLED_WINDOW(lookup_widget(appwindow, "rxscrolledwindow"));
	freqspinbutton = GTK_SPIN_BUTTON(lookup_widget(appwindow, "freqspinbutton"));
	metricdial = GTK_DIAL(lookup_widget(appwindow, "metricdial"));
	rxtextview = GTK_TEXT_VIEW(lookup_widget(appwindow, "rxtext"));
	txtextview = GTK_TEXT_VIEW(lookup_widget(appwindow, "txtext"));
	rxbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(rxtextview));
	txbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(txtextview));

	statusbar = GTK_STATUSBAR(lookup_widget(appwindow, "mainstatusbar"));
	uiinfo = (GnomeUIInfo *) gtk_object_get_data(GTK_OBJECT(appwindow), "menubar1_uiinfo");
	if (uiinfo)
		gnome_app_install_statusbar_menu_hints(statusbar, uiinfo);

	rxtag = gtk_text_buffer_create_tag(rxbuffer, "rxtag", NULL);
	txtag = gtk_text_buffer_create_tag(rxbuffer, "txtag", NULL);
	hltag = gtk_text_buffer_create_tag(rxbuffer, "hltag", NULL);

	tag = gtk_text_buffer_create_tag(rxbuffer, "qsodatatag", NULL);
	g_object_set(G_OBJECT(tag), "background", "grey", NULL);

	/* load config files and configure things */
	conf_init();
	macroconfig_load();

	conf_set_ptt();

	/* set word wrap for rx window */
	gtk_text_view_set_wrap_mode(rxtextview, GTK_WRAP_WORD);

	/* create end marks for tx and rx buffers */
	gtk_text_buffer_get_end_iter(txbuffer, &iter);
	gtk_text_buffer_create_mark(txbuffer, "end", &iter, FALSE);
	gtk_text_buffer_get_end_iter(rxbuffer, &iter);
	gtk_text_buffer_create_mark(rxbuffer, "end", &iter, FALSE);

	/* periodic task - main loop */
	gtk_timeout_add(100, main_loop, NULL);

	/* periodic task - update clock */
	gtk_timeout_add(1000, clock_cb, NULL);

	/* show the main window */
	gtk_widget_show(appwindow);

	statusbar_set_mode(0);

	/* initialize the trx thread */
	trx_init();
	waterfall_set_samplerate(waterfall, 8000);
	waterfall_set_frequency(waterfall, 1000.0);
	waterfall_set_center_frequency(waterfall, 1500.0);
	waterfall_set_bandwidth(waterfall, trx_get_bandwidth());

	/* let GTK do it's job */
	gdk_threads_enter();
	gtk_main ();
	gdk_threads_leave();

	conf_clear();
	log_to_file_activate(FALSE);

	return 0;
}

